/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.modules.parking.service.impl;

import admin.annotation.CurDataSource;
import admin.base.PageInfo;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.modules.parking.domain.ParkingRoadInfo;
import admin.modules.parking.service.ParkingRoadInfoService;
import admin.modules.parking.service.dto.ParkingRoadInfoDto;
import admin.modules.parking.service.dto.ParkingRoadInfoQueryCriteria;
import admin.modules.parking.service.mapper.ParkingRoadInfoMapper;
import admin.utils.ConvertUtil;
import admin.utils.FileUtil;
import admin.utils.PageUtil;
import admin.utils.ValidationUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.google.common.base.Stopwatch;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
* @website https://el-admin.vip
* @description 服务实现
* @author wl
* @date 2020-07-12
**/
@Service
@Slf4j
@RequiredArgsConstructor
public class ParkingRoadInfoServiceImpl extends CommonServiceImpl<ParkingRoadInfoMapper, ParkingRoadInfo> implements ParkingRoadInfoService {

    private final ParkingRoadInfoMapper parkingRoadInfoMapper;

    @Override
    public PageInfo<ParkingRoadInfoDto> queryAll(ParkingRoadInfoQueryCriteria criteria, Pageable pageable){
        IPage<ParkingRoadInfo> page = PageUtil.toMybatisPage(pageable, false);
        IPage<ParkingRoadInfo> pageList = parkingRoadInfoMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertPage(pageList, ParkingRoadInfoDto.class);
    }

    @Override
    public List<ParkingRoadInfoDto> queryAll(ParkingRoadInfoQueryCriteria criteria){
        return ConvertUtil.convertList(parkingRoadInfoMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)), ParkingRoadInfoDto.class);
    }

    @Override
    @Transactional
    public ParkingRoadInfoDto findById(Long roadId) {
        ParkingRoadInfo parkingRoadInfo = parkingRoadInfoMapper.selectById(roadId);
        ValidationUtil.isNull(parkingRoadInfo.getRoadId(),"ParkingRoadInfo","roadId",roadId);
        return ConvertUtil.convert(parkingRoadInfo, ParkingRoadInfoDto.class);
    }

    @Override
    public ParkingRoadInfo findByAreaIdAndRoadSerial(String areaId, String roadSerial) {
        ParkingRoadInfo parkingRoadInfo = parkingRoadInfoMapper.findByAreaIdAndRoadSerial(areaId, roadSerial);
        return parkingRoadInfo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ParkingRoadInfoDto create(ParkingRoadInfo resources) {
        parkingRoadInfoMapper.insert(resources);
        return ConvertUtil.convert(resources, ParkingRoadInfoDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(ParkingRoadInfo resources) {
        ParkingRoadInfo parkingRoadInfo = parkingRoadInfoMapper.selectById(resources.getRoadId());
        ValidationUtil.isNull( parkingRoadInfo.getRoadId(),"ParkingRoadInfo","id",resources.getRoadId());
        parkingRoadInfo.copy(resources);
        parkingRoadInfoMapper.updateById(parkingRoadInfo);
    }

    @Override
    public void deleteAll(Long[] ids) {
        for (Long roadId : ids) {
            parkingRoadInfoMapper.deleteById(roadId);
        }
    }

    @Override
    public void download(List<ParkingRoadInfoDto> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (ParkingRoadInfoDto parkingRoadInfo : all) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("路段名称", parkingRoadInfo.getRoadName());
            map.put("路段编号", parkingRoadInfo.getRoadSerial());
            map.put("省id", parkingRoadInfo.getProvinceId());
            map.put(" provinceName",  parkingRoadInfo.getProvinceName());
            map.put(" cityId",  parkingRoadInfo.getCityId());
            map.put(" cityName",  parkingRoadInfo.getCityName());
            map.put(" areaId",  parkingRoadInfo.getAreaId());
            map.put(" areaName",  parkingRoadInfo.getAreaName());
            map.put(" address",  parkingRoadInfo.getAddress());
            map.put(" latitude",  parkingRoadInfo.getLatitude());
            map.put(" longitude",  parkingRoadInfo.getLongitude());
            map.put("路段状态  1：启用，2：禁用", parkingRoadInfo.getStatus());
            map.put(" createTime",  parkingRoadInfo.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }


    @Override
    public Integer parkingInfoCount(Long roadId, Integer status) {
        return parkingRoadInfoMapper.parkingInfoCount(roadId,status);
    }

    @Override
    @CurDataSource("slaver")
    public List<ParkingRoadInfoDto> getRoadInfo() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        //1.获取所有路段信息
        List<ParkingRoadInfoDto> roadInfoList = queryAll(null);

        //2.归集
        roadInfoList.forEach(i->{
            i.setRemainNum(0);
        });
        log.info("查询路段剩余泊位数耗时{}毫秒",stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return roadInfoList;
    }
}
