/*
 *  Copyright 2019-2020 Zheng Jie
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package admin.modules.system.service.impl;

import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.exception.BadRequestException;
import admin.exception.EntityExistException;
import admin.model.menu.ShowNavModel;
import admin.modules.system.domain.Menu;
import admin.modules.system.domain.Role;
import admin.modules.system.domain.User;
import admin.modules.system.domain.vo.MenuMetaVo;
import admin.modules.system.domain.vo.MenuVo;
import admin.modules.system.service.MenuService;
import admin.modules.system.service.RoleService;
import admin.modules.system.service.dto.MenuDto;
import admin.modules.system.service.dto.MenuQueryCriteria;
import admin.modules.system.service.dto.RoleSmallDto;
import admin.modules.system.service.mapper.MenuMapper;
import admin.modules.system.service.mapper.UserMapper;
import admin.utils.*;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.RequiredArgsConstructor;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 */
@Service
@RequiredArgsConstructor
@CacheConfig(cacheNames = "menu")
public class MenuServiceImpl extends CommonServiceImpl<MenuMapper, Menu> implements MenuService {

    private final UserMapper userMapper;
    private final MenuMapper menuMapper;
    private final RoleService roleService;
    private final RedisUtils redisUtils;

    @Override
    public List<MenuDto> queryAll(MenuQueryCriteria criteria, Boolean isQuery) throws Exception {
        if(isQuery){
            criteria.setPidIsNull(true);
            List<Field> fields = QueryHelpMybatisPlus.getAllFields(criteria.getClass(), new ArrayList<>());
            for (Field field : fields) {
                //设置对象的访问权限，保证对private的属性的访问
                field.setAccessible(true);
                Object val = field.get(criteria);
                if("pidIsNull".equals(field.getName())){
                    continue;
                }
                if (ObjectUtil.isNotNull(val)) {
                    criteria.setPidIsNull(null);
                    break;
                }
            }
        }
        QueryWrapper<Menu> predicate = QueryHelpMybatisPlus.getPredicate(criteria);
        predicate.orderByAsc("menu_sort");
        List<Menu> menus = menuMapper.selectList(predicate);
        return ConvertUtil.convertList(menus, MenuDto.class);
    }

    @Override
//    @Cacheable(key = "'id:' + #p0")
    public MenuDto findById(long id) {
        Menu menu = menuMapper.selectById(id);
        ValidationUtil.isNull(menu.getId(),"Menu","id",id);
        return ConvertUtil.convert(menu, MenuDto.class);
    }

    /**
     * 用户角色改变时需清理缓存
     * @param currentUserId /
     * @return /
     */
    @Override
//    @Cacheable(key = "'user:' + #p0")
    public List<MenuDto> findByUser(Long currentUserId) {
        List<RoleSmallDto> roles = roleService.findByUsersId(currentUserId);
        Set<Long> roleIds = roles.stream().map(RoleSmallDto::getId).collect(Collectors.toSet());
        LinkedHashSet<Menu> menus = menuMapper.findByRoleIdsAndTypeNot(roleIds, 2);
        return menus.stream().map(menu -> ConvertUtil.convert(menu, MenuDto.class)).collect(Collectors.toList());
    }

    @Override
    public List<ShowNavModel> getShowNav() {
        MenuQueryCriteria criteria = new MenuQueryCriteria();
        List<MenuDto> menuDtos = ConvertUtil.convertList(menuMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)), MenuDto.class);
        criteria.setShowNav(1);
        List<MenuDto> NavMenuDtos = ConvertUtil.convertList(menuMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)), MenuDto.class);
        List<ShowNavModel> showNavModelList = new ArrayList<>();
        for(MenuDto menuDto:NavMenuDtos){
            String path = "";
            Integer isLink = 0;
            if(menuDto.getIFrame()){
                path = menuDto.getPath();
                isLink = 1;
            }else{
                List<MenuDto> parentMenu = new ArrayList<>();
                getParentNode(parentMenu,menuDtos,menuDto.getId().toString());
                for(MenuDto menuDto1:parentMenu){
                    String temp = menuDto1.getPath().startsWith("/")?menuDto1.getPath():"/"+menuDto1.getPath();
                    path+=temp;
                }
                int count = 0;
                String str = "";
                str = path;
                while(str.indexOf("/") != -1) {
                    str = str.substring(str.indexOf("/") + 1,str.length());
                    count++;
                }
                if(count>=4){
                    path = path.substring(1);
                    path = path.substring(path.indexOf("/"));
                }
            }
            ShowNavModel showNavModel = new ShowNavModel();
            showNavModel.setId(menuDto.getId());
            showNavModel.setShowNav(menuDto.getShowNav());
            showNavModel.setNavQuick(menuDto.getNavQuick());
            showNavModel.setTitle(menuDto.getTitle());
            showNavModel.setPath(path);
            showNavModel.setIsLink(isLink);
            showNavModelList.add(showNavModel);
        }
        return showNavModelList;
    }

    private static void getParentNode(List<MenuDto> parentMenu, List<MenuDto> menuList, String id) {
        menuList.stream()
                .filter(menu -> StringUtils.isNotBlank(id) && id.equals(menu.getId().toString()))
                .forEach(menu -> {
                    if(menu.getPid()!=null) {
                        getParentNode(parentMenu, menuList, menu.getPid().toString());
                    }
                    parentMenu.add(menu);
                });

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateQuickNav(Set<Long> ids) {
        //清空所有快捷方式
        menuMapper.delQuick();
        menuMapper.addQuick(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(Menu resources) {
        if(menuMapper.findByTitle(resources.getTitle()) != null){
            throw new EntityExistException(Menu.class,"title",resources.getTitle());
        }
        if(StringUtils.isNotBlank(resources.getComponentName())){
            if(menuMapper.findByComponentName(resources.getComponentName()) != null){
                throw new EntityExistException(Menu.class,"componentName",resources.getComponentName());
            }
        }
        if(resources.getPid().equals(0L)){
            resources.setPid(null);
        }
        if(resources.getIFrame()){
            String http = "http://", https = "https://";
            if (!(resources.getPath().toLowerCase().startsWith(http)||resources.getPath().toLowerCase().startsWith(https))) {
                throw new BadRequestException("外链必须以http://或者https://开头");
            }
        }
        menuMapper.insert(resources);
        // 计算子节点数目
        resources.setSubCount(0);
        // 更新父节点菜单数目
        updateSubCnt(resources.getPid());
        redisUtils.del("menu::pid:" + (resources.getPid() == null ? 0 : resources.getPid()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(Menu resources) {
        if(resources.getId().equals(resources.getPid())) {
            throw new BadRequestException("上级不能为自己");
        }
        Menu menu = menuMapper.selectById(resources.getId());
        ValidationUtil.isNull(menu.getId(),"Permission","id",resources.getId());

        if(resources.getIFrame()){
            String http = "http://", https = "https://";
            if (!(resources.getPath().toLowerCase().startsWith(http)||resources.getPath().toLowerCase().startsWith(https))) {
                throw new BadRequestException("外链必须以http://或者https://开头");
            }
        }
        Menu menu1 = menuMapper.findByTitle(resources.getTitle());

        if(menu1 != null && !menu1.getId().equals(menu.getId())){
            throw new EntityExistException(Menu.class,"title",resources.getTitle());
        }

        if(resources.getPid().equals(0L)){
            resources.setPid(null);
        }

        // 记录的父节点ID
        Long oldPid = menu.getPid();
        Long newPid = resources.getPid();

        if(StringUtils.isNotBlank(resources.getComponentName())){
            menu1 = menuMapper.findByComponentName(resources.getComponentName());
            if(menu1 != null && !menu1.getId().equals(menu.getId())){
                throw new EntityExistException(Menu.class,"componentName",resources.getComponentName());
            }
        }
        menu.setTitle(resources.getTitle());
        menu.setComponent(resources.getComponent());
        menu.setPath(resources.getPath());
        menu.setIcon(resources.getIcon());
        menu.setIFrame(resources.getIFrame());
        menu.setPid(resources.getPid());
        menu.setMenuSort(resources.getMenuSort());
        menu.setCache(resources.getCache());
        menu.setHidden(resources.getHidden());
        menu.setComponentName(resources.getComponentName());
        menu.setPermission(resources.getPermission());
        menu.setType(resources.getType());
        menu.setShowNav(resources.getShowNav());
        if(resources.getShowNav()==0){
            menu.setNavQuick(0);
        }
        menuMapper.updateById(menu);
        // 计算父级菜单节点数目
        updateSubCnt(oldPid);
        updateSubCnt(newPid);
        // 清理缓存
        delCaches(resources.getId(), oldPid, newPid);
    }

    @Override
    public Set<Menu> getDeleteMenus(List<Menu> menuList, Set<Menu> menuSet) {
        // 递归找出待删除的菜单
        for (Menu menu1 : menuList) {
            menuSet.add(menu1);
            List<Menu> menus = menuMapper.findByPid(menu1.getId());
            if(menus!=null && menus.size()!=0){
                getDeleteMenus(menus, menuSet);
            }
        }
        return menuSet;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Set<Menu> menuSet) {
        for (Menu menu : menuSet) {
            // 清理缓存
            delCaches(menu.getId(), menu.getPid(), null);
            roleService.untiedMenu(menu.getId());
            menuMapper.deleteById(menu.getId());
            updateSubCnt(menu.getPid());
        }
    }

    @Override
//    @Cacheable(key = "'pid:' + #p0")
    public List<MenuDto> getMenus(Long pid) {
        List<Menu> menus;
        if (pid != null && !pid.equals(0L)) {
            menus = lambdaQuery().eq(Menu::getPid, pid).orderByAsc(Menu::getMenuSort).list();
        } else {
            menus = lambdaQuery().isNull(Menu::getPid).orderByAsc(Menu::getMenuSort).list();
        }
        return ConvertUtil.convertList(menus, MenuDto.class);
    }

    @Override
    public List<MenuDto> getSuperior(MenuDto menuDto, List<Menu> menus) {
        if(menuDto.getPid() == null){
            menus.addAll(menuMapper.findByPidIsNull());
            return ConvertUtil.convertList(menus, MenuDto.class);
        }
        menus.addAll(menuMapper.findByPid(menuDto.getPid()));
        return getSuperior(findById(menuDto.getPid()), menus);
    }

    @Override
    public List<MenuDto> buildTree(List<MenuDto> menuDtos) {
        List<MenuDto> trees = new ArrayList<>();
        Set<Long> ids = new HashSet<>();
        for (MenuDto menuDTO : menuDtos) {
            if (menuDTO.getPid() == null) {
                trees.add(menuDTO);
            }
            for (MenuDto it : menuDtos) {
                if (menuDTO.getId().equals(it.getPid())) {
                    if (menuDTO.getChildren() == null) {
                        menuDTO.setChildren(new ArrayList<>());
                    }
                    menuDTO.getChildren().add(it);
                    ids.add(it.getId());
                }
            }
        }
        if(trees.size() == 0){
            trees = menuDtos.stream().filter(s -> !ids.contains(s.getId())).collect(Collectors.toList());
        }
        return trees;
    }

    @Override
    public List<MenuVo> buildMenus(List<MenuDto> menuDtos) {
        List<MenuVo> list = new LinkedList<>();
        menuDtos.forEach(menuDTO -> {
                    if (menuDTO!=null){
                        List<MenuDto> menuDtoList = menuDTO.getChildren();
                        MenuVo menuVo = new MenuVo();
                        menuVo.setName(ObjectUtil.isNotEmpty(menuDTO.getComponentName())  ? menuDTO.getComponentName() : menuDTO.getTitle());
                        // 一级目录需要加斜杠，不然会报警告
                        menuVo.setPath(menuDTO.getPid() == null ? "/" + menuDTO.getPath() :menuDTO.getPath());
                        menuVo.setHidden(menuDTO.getHidden());
                        menuVo.setIFrame(menuDTO.getIFrame());
                        // 如果不是外链
                        if(!menuDTO.getIFrame()){
                            if(menuDTO.getPid() == null){
                                menuVo.setComponent(StrUtil.isEmpty(menuDTO.getComponent())?"Layout":menuDTO.getComponent());
                            }else if(!StrUtil.isEmpty(menuDTO.getComponent())){
                                menuVo.setComponent(menuDTO.getComponent());
                            }
                        }
                        menuVo.setMeta(new MenuMetaVo(menuDTO.getTitle(),menuDTO.getIcon(),!menuDTO.getCache()));
                        if(menuDtoList !=null && menuDtoList.size()!=0){
                            menuVo.setAlwaysShow(true);
                            menuVo.setRedirect("noredirect");
                            menuVo.setChildren(buildMenus(menuDtoList));
                            // 处理是一级菜单并且没有子菜单的情况
                        } else if(menuDTO.getPid() == null){
                            MenuVo menuVo1 = new MenuVo();
                            menuVo1.setMeta(menuVo.getMeta());
                            // 非外链
                            if(!menuDTO.getIFrame()){
                                menuVo1.setPath("index");
                                menuVo1.setName(menuVo.getName());
                                menuVo1.setComponent(menuVo.getComponent());
                            } else {
                                menuVo1.setPath(menuDTO.getPath());
                            }
                            menuVo.setName(null);
                            menuVo.setMeta(null);
                            menuVo.setComponent("Layout");
                            List<MenuVo> list1 = new ArrayList<>();
                            list1.add(menuVo1);
                            menuVo.setChildren(list1);
                        }
                        list.add(menuVo);
                    }
                }
        );
        return list;
    }

    @Override
    public Menu findOne(Long id) {
        Menu menu = menuMapper.selectById(id);
        ValidationUtil.isNull(menu.getId(),"Menu","id",id);
        return menu;
    }

    @Override
    public void download(List<MenuDto> menuDtos, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (MenuDto menuDTO : menuDtos) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("菜单标题", menuDTO.getTitle());
            map.put("菜单类型", menuDTO.getType() == null ? "目录" : menuDTO.getType() == 1 ? "菜单" : "按钮");
            map.put("权限标识", menuDTO.getPermission());
            map.put("外链菜单", menuDTO.getIFrame() ? "是" : "否");
            map.put("菜单可见", menuDTO.getHidden() ? "否" : "是");
            map.put("是否缓存", menuDTO.getCache() ? "是" : "否");
            map.put("创建日期", menuDTO.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }

    private void updateSubCnt(Long menuId){
        if(menuId != null){
            int count = menuMapper.countByPid(menuId);
            menuMapper.updateSubCntById(count, menuId);
        }
    }

    /**
     * 清理缓存
     * @param id 菜单ID
     * @param oldPid 旧的菜单父级ID
     * @param newPid 新的菜单父级ID
     */
    public void delCaches(Long id, Long oldPid, Long newPid){
        List<User> users = userMapper.findByMenuId(id);
        redisUtils.del("menu::id:" +id);
        redisUtils.delByKeys("menu::user:",users.stream().map(User::getId).collect(Collectors.toSet()));
        redisUtils.del("menu::pid:" + (oldPid == null ? 0 : oldPid));
        redisUtils.del("menu::pid:" + (newPid == null ? 0 : newPid));
        // 清除 Role 缓存
        List<Role> roles = roleService.findInMenuId(new ArrayList<Long>(){{
            add(id);
            add(newPid == null ? 0 : newPid);
        }});
        redisUtils.delByKeys("role::id:",roles.stream().map(Role::getId).collect(Collectors.toSet()));
    }
}
