/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.modules.system.service.impl;

import admin.base.PageInfo;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.modules.system.domain.SysContent;
import admin.modules.system.service.SysContentService;
import admin.modules.system.service.dto.SysContentDto;
import admin.modules.system.service.dto.SysContentQueryCriteria;
import admin.modules.system.service.mapper.SysContentMapper;
import admin.utils.ConvertUtil;
import admin.utils.FileUtil;
import admin.utils.PageUtil;
import admin.utils.ValidationUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
* @website https://el-admin.vip
* @description 服务实现
* @author lj
* @date 2023-06-01
**/
@Service
@RequiredArgsConstructor
public class SysContentServiceImpl extends CommonServiceImpl<SysContentMapper, SysContent> implements SysContentService {

    private final SysContentMapper sysContentMapper;

    @Override
    public PageInfo<SysContentDto> queryAll(SysContentQueryCriteria criteria, Pageable pageable){
        IPage<SysContent> page = PageUtil.toMybatisPage(pageable, false);
        IPage<SysContent> pageList = sysContentMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertPage(pageList, SysContentDto.class);
    }

    @Override
    public List<SysContentDto> queryAll(SysContentQueryCriteria criteria){
        List<SysContent> list = sysContentMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertList(list, SysContentDto.class);
    }

    @Override
    @Transactional
    public SysContentDto findById(Long contentId) {
        SysContent sysContent = sysContentMapper.selectById(contentId);
        ValidationUtil.isNull(sysContent.getContentId(),"SysContent","contentId",contentId);
        return ConvertUtil.convert(sysContent, SysContentDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SysContentDto create(SysContent resources) {
        sysContentMapper.insert(resources);
        return ConvertUtil.convert(resources, SysContentDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SysContent resources) {
        SysContent sysContent = sysContentMapper.selectById(resources.getContentId());
        ValidationUtil.isNull( sysContent.getContentId(),"SysContent","id",resources.getContentId());
        sysContent.copy(resources);
        sysContentMapper.updateById(sysContent);
    }

    @Override
    public void deleteAll(Long[] ids) {
        for (Long contentId : ids) {
            sysContentMapper.deleteById(contentId);
        }
    }

    @Override
    public void download(List<SysContentDto> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (SysContentDto sysContent : all) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("标题", sysContent.getTitle());
            map.put("目录id", sysContent.getDirectoryId());
            map.put("内容", sysContent.getContent());
            map.put("创建时间", sysContent.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
