/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.modules.system.service.impl;

import admin.base.PageInfo;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.modules.system.domain.SysDirectory;
import admin.modules.system.service.SysDirectoryService;
import admin.modules.system.service.dto.SysDirectoryDto;
import admin.modules.system.service.dto.SysDirectoryQueryCriteria;
import admin.modules.system.service.mapper.SysDirectoryMapper;
import admin.utils.ConvertUtil;
import admin.utils.FileUtil;
import admin.utils.PageUtil;
import admin.utils.ValidationUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
* @website https://el-admin.vip
* @description 服务实现
* @author lj
* @date 2023-06-01
**/
@Service
@RequiredArgsConstructor
public class SysDirectoryServiceImpl extends CommonServiceImpl<SysDirectoryMapper, SysDirectory> implements SysDirectoryService {

    private final SysDirectoryMapper sysDirectoryMapper;

    @Override
    public PageInfo<SysDirectoryDto> queryAll(SysDirectoryQueryCriteria criteria, Pageable pageable){
        IPage<SysDirectory> page = PageUtil.toMybatisPage(pageable, false);
        IPage<SysDirectory> pageList = sysDirectoryMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertPage(pageList, SysDirectoryDto.class);
    }

    @Override
    public List<SysDirectoryDto> queryAll(SysDirectoryQueryCriteria criteria){
        List<SysDirectory> list = sysDirectoryMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertList(list, SysDirectoryDto.class);
    }

    @Override
    @Transactional
    public SysDirectoryDto findById(Long directoryId) {
        SysDirectory sysDirectory = sysDirectoryMapper.selectById(directoryId);
        ValidationUtil.isNull(sysDirectory.getDirectoryId(),"SysDirectory","directoryId",directoryId);
        return ConvertUtil.convert(sysDirectory, SysDirectoryDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SysDirectoryDto create(SysDirectory resources) {
        sysDirectoryMapper.insert(resources);
        return ConvertUtil.convert(resources, SysDirectoryDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SysDirectory resources) {
        SysDirectory sysDirectory = sysDirectoryMapper.selectById(resources.getDirectoryId());
        ValidationUtil.isNull( sysDirectory.getDirectoryId(),"SysDirectory","id",resources.getDirectoryId());
        sysDirectory.copy(resources);
        sysDirectoryMapper.updateById(sysDirectory);
    }

    @Override
    public void deleteAll(Long[] ids) {
        for (Long directoryId : ids) {
            sysDirectoryMapper.deleteById(directoryId);
        }
    }

    @Override
    public void download(List<SysDirectoryDto> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (SysDirectoryDto sysDirectory : all) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("顶级父类为0", sysDirectory.getParentId());
            map.put("目录名称", sysDirectory.getName());
            map.put("创建时间", sysDirectory.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
