/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.modules.system.service.impl;

import admin.base.PageInfo;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.modules.system.domain.SysUserNotice;
import admin.modules.system.service.SysUserNoticeService;
import admin.modules.system.service.dto.SysUserNoticeDto;
import admin.modules.system.service.dto.SysUserNoticeQueryCriteria;
import admin.modules.system.service.mapper.SysUserNoticeMapper;
import admin.utils.ConvertUtil;
import admin.utils.FileUtil;
import admin.utils.PageUtil;
import admin.utils.ValidationUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
* @website https://el-admin.vip
* @description 服务实现
* @author wl
* @date 2020-07-15
**/
@Service
@RequiredArgsConstructor
public class SysUserNoticeServiceImpl extends CommonServiceImpl<SysUserNoticeMapper, SysUserNotice> implements SysUserNoticeService {

    private final SysUserNoticeMapper sysUserNoticeMapper;

    @Override
    public Integer getUnReadMsgCount(Integer noticeType,Long jobId) {
        return sysUserNoticeMapper.countUnReadNotice(noticeType,jobId);
    }

    @Override
    public PageInfo<SysUserNoticeDto> queryAll(SysUserNoticeQueryCriteria criteria, Pageable pageable){
        IPage<SysUserNotice> page = PageUtil.toMybatisPage(pageable, false);
        IPage<SysUserNotice> pageList = sysUserNoticeMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertPage(pageList, SysUserNoticeDto.class);
    }

    @Override
    public List<SysUserNoticeDto> queryAll(SysUserNoticeQueryCriteria criteria){
        return ConvertUtil.convertList(sysUserNoticeMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)),SysUserNoticeDto.class);
    }

    @Override
    @Transactional
    public SysUserNoticeDto findById(Long noticeId) {
        SysUserNotice sysUserNotice = sysUserNoticeMapper.selectById(noticeId);
        ValidationUtil.isNull(sysUserNotice.getNoticeId(),"SysUserNotice","noticeId",noticeId);
        return ConvertUtil.convert(sysUserNotice, SysUserNoticeDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SysUserNoticeDto create(SysUserNotice resources) {
        sysUserNoticeMapper.insert(resources);
        return ConvertUtil.convert(resources, SysUserNoticeDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SysUserNotice resources) {
        SysUserNotice sysUserNotice = sysUserNoticeMapper.selectById(resources.getNoticeId());
        ValidationUtil.isNull( sysUserNotice.getNoticeId(),"SysUserNotice","id",resources.getNoticeId());
        sysUserNotice.copy(resources);
        sysUserNoticeMapper.updateById(sysUserNotice);
    }

    @Override
    public void deleteAll(Long[] ids) {
        for (Long noticeId : ids) {
            sysUserNoticeMapper.deleteById(noticeId);
        }
    }


    @Override
    public void download(List<SysUserNoticeDto> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (SysUserNoticeDto sysUserNotice : all) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("通知类型：1入场 2离场3代办4警告", sysUserNotice.getNoticeType());
            map.put("是否已读 1：未读，2：已读", sysUserNotice.getIsRead());
            map.put(" plateId",  sysUserNotice.getPlateId());
            map.put("设备id", sysUserNotice.getDevId());
            map.put("是否黑车  1：是，2：否", sysUserNotice.getIsBlack());
            map.put(" parkingId",  sysUserNotice.getParkingId());
            map.put("岗位id", sysUserNotice.getJobId());
            map.put("创建时间", sysUserNotice.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
