/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.modules.system.service.impl;

import admin.base.PageInfo;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.modules.security.config.bean.SecurityProperties;
import admin.modules.security.service.OnlineUserService;
import admin.modules.system.domain.SystemOpLog;
import admin.modules.system.service.SystemOpLogService;
import admin.modules.system.service.dto.SystemOpLogDto;
import admin.modules.system.service.dto.SystemOpLogQueryCriteria;
import admin.modules.system.service.mapper.SystemOpLogMapper;
import admin.utils.*;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
* @website https://el-admin.vip
* @description 服务实现
* @author wl
* @date 2020-07-19
**/
@Service
@RequiredArgsConstructor
@Slf4j
public class SystemOpLogServiceImpl extends CommonServiceImpl<SystemOpLogMapper, SystemOpLog> implements SystemOpLogService {

    private final SystemOpLogMapper systemOpLogMapper;
    private final OnlineUserService onlineUserService;
    private final SecurityProperties properties;


    @Override
    public void insertSystemOpLogUnLogin(Long orderId, String msg, String userName) {
        try {
            SystemOpLog systemOpLog = new SystemOpLog();
            systemOpLog.setOrderId(orderId);
            systemOpLog.setCreateTime(DateUtil.getNowTimestamp());
            systemOpLog.setLogMsg(msg.length()>1000?msg.substring(0,1000):msg);
            systemOpLog.setUserName(userName);
            create(systemOpLog);
        }catch (Exception ex){
            log.error("记录未登录状态下系统日志异常{},{}",orderId,msg,ex);
        }
    }

    @Override
    public PageInfo<SystemOpLogDto> queryAll(SystemOpLogQueryCriteria criteria, Pageable pageable){
        IPage<SystemOpLog> page = PageUtil.toMybatisPage(pageable, false);
        IPage<SystemOpLog> pageList = systemOpLogMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        return ConvertUtil.convertPage(pageList, SystemOpLogDto.class);
    }

    @Override
    public List<SystemOpLogDto> queryAll(SystemOpLogQueryCriteria criteria){
        return ConvertUtil.convertList(systemOpLogMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)), SystemOpLogDto.class);
    }

    @Override
    @Transactional
    public SystemOpLogDto findById(Long systemOpLogId) {
        SystemOpLog systemOpLog = systemOpLogMapper.selectById(systemOpLogId);
        ValidationUtil.isNull(systemOpLog.getSystemOpLogId(),"SystemOpLog","systemOpLogId",systemOpLogId);
        return ConvertUtil.convert(systemOpLog, SystemOpLogDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SystemOpLogDto create(SystemOpLog resources) {
        systemOpLogMapper.insert(resources);
        return ConvertUtil.convert(resources, SystemOpLogDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SystemOpLog resources) {
        SystemOpLog systemOpLog = systemOpLogMapper.selectById(resources.getSystemOpLogId());
        ValidationUtil.isNull( systemOpLog.getSystemOpLogId(),"SystemOpLog","id",resources.getSystemOpLogId());
        systemOpLog.copy(resources);
        systemOpLogMapper.updateById(systemOpLog);
    }

    @Override
    public void deleteAll(Long[] ids) {
        for (Long systemOpLogId : ids) {
            systemOpLogMapper.deleteById(systemOpLogId);
        }
    }

    @Override
    public void download(List<SystemOpLogDto> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (SystemOpLogDto systemOpLog : all) {
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("操作人", systemOpLog.getUserName());
            map.put("日志内容", systemOpLog.getLogMsg());
            map.put("创建时间", systemOpLog.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
