/*
 *  Copyright 2019-2020 Zheng Jie
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package admin.modules.system.service.impl;

import admin.annotation.CurDataSource;
import admin.base.QueryHelpMybatisPlus;
import admin.base.impl.CommonServiceImpl;
import admin.config.FileProperties;
import admin.exception.EntityExistException;
import admin.exception.EntityNotFoundException;
import admin.modules.security.service.UserCacheClean;
import admin.modules.system.domain.*;
import admin.modules.system.service.DeptService;
import admin.modules.system.service.SysUsersRolesService;
import admin.modules.system.service.UserService;
import admin.modules.system.service.dto.*;
import admin.modules.system.service.mapper.JobMapper;
import admin.modules.system.service.mapper.RoleMapper;
import admin.modules.system.service.mapper.UserMapper;
import admin.utils.*;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotBlank;
import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @date 2018-11-23
 */
@Service
@Slf4j
@CurDataSource("master")
@RequiredArgsConstructor
@CacheConfig(cacheNames = "user")
public class UserServiceImpl extends CommonServiceImpl<UserMapper, User> implements UserService {

    private final UserMapper userMapper;
    private final FileProperties properties;
    private final RedisUtils redisUtils;
    private final UserCacheClean userCacheClean;
    private final DeptService deptService;
    private final SysUsersRolesService sysUsersRolesService;
    private final RoleMapper roleMapper;
    private final JobMapper jobMapper;

    @Override
    public Object queryAll(UserQueryCriteria criteria, Pageable pageable) {
        IPage<User> page = PageUtil.toMybatisPage(pageable, false);
        IPage<User> pageList = userMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        if(pageList.getRecords().size()>0) {
            Map<Long, Dept> deptMap = deptService.list().parallelStream()
                    .collect(Collectors.toMap(Dept::getId, Function.identity(), (x, y) -> x));

            Set<Long> userIds = pageList.getRecords().stream().map(User::getId).collect(Collectors.toSet());

            Map<Long, Set<SysUsersRoles>> usersRolesMap = sysUsersRolesService.lambdaQuery()
                    .in(SysUsersRoles::getUserId, userIds)
                    .list()
                    .stream()
                    .collect(Collectors.groupingBy(SysUsersRoles::getUserId, Collectors.toSet()));

            Map<Long, Role> roleMap = roleMapper.selectList(null)
                    .stream()
                    .collect(Collectors.toMap(Role::getId, Function.identity()));

            Map<Long, Job> jobMap = jobMapper.selectList(null)
                    .stream()
                    .collect(Collectors.toMap(Job::getId, Function.identity()));

            pageList.getRecords().forEach(user -> {
                user.setDept(deptMap.get(user.getDeptId()));
                if (usersRolesMap.containsKey(user.getId())) {
                    user.setRoles(usersRolesMap.get(user.getId()).stream().map(ur -> {
                        Role role = roleMap.get(ur.getRoleId());
                        return role;
                    }).collect(Collectors.toSet()));
                }
            });
        }
        return ConvertUtil.convertPage(pageList, UserDto.class);
    }

    @Override
    public Object queryAllBusiness(UserInfoQueryCriteria criteria, Pageable pageable) {
        IPage<User> page = PageUtil.toMybatisPage(pageable, false);
        IPage<User> pageList = userMapper.selectPage(page, QueryHelpMybatisPlus.getPredicate(criteria));
        if(pageList.getRecords().size()>0) {
            Map<Long, Dept> deptMap = deptService.list().parallelStream()
                    .collect(Collectors.toMap(Dept::getId, Function.identity(), (x, y) -> x));

            Set<Long> userIds = pageList.getRecords().stream().map(User::getId).collect(Collectors.toSet());

            Map<Long, Set<SysUsersRoles>> usersRolesMap = sysUsersRolesService.lambdaQuery()
                    .in(SysUsersRoles::getUserId, userIds)
                    .list()
                    .stream()
                    .collect(Collectors.groupingBy(SysUsersRoles::getUserId, Collectors.toSet()));

            Map<Long, Role> roleMap = roleMapper.selectList(null)
                    .stream()
                    .collect(Collectors.toMap(Role::getId, Function.identity()));

            Map<Long, Job> jobMap = jobMapper.selectList(null)
                    .stream()
                    .collect(Collectors.toMap(Job::getId, Function.identity()));

            pageList.getRecords().forEach(user -> {
                user.setDept(deptMap.get(user.getDeptId()));
                if (usersRolesMap.containsKey(user.getId())) {
                    user.setRoles(usersRolesMap.get(user.getId()).stream().map(ur -> {
                        Role role = roleMap.get(ur.getRoleId());
                        return role;
                    }).collect(Collectors.toSet()));
                }
            });
        }
        return ConvertUtil.convertPage(pageList, UserDto.class);
    }

    @Override
    public List<UserDto> queryAll(UserQueryCriteria criteria) {
        return ConvertUtil.convertList(userMapper.selectList(QueryHelpMybatisPlus.getPredicate(criteria)),UserDto.class);
    }

    @Override
    public List<UserDto> allUser() {
        List<Map<String,Object>> maps= userMapper.allUserData();
        List<UserDto> userDtos=new ArrayList<>();
        maps.forEach(v->{
            if(v!=null){
                try {
                    UserDto userDto = new UserDto();
                    userDto.setId(Long.parseLong(v.get("userId").toString()));
                    userDto.setUsername(v.get("userName")==null?"":v.get("userName").toString());
                    userDto.setPhone(v.get("phone")==null?"":v.get("phone").toString());
                    userDto.setIsBusiness(v.get("isBusiness")==null?0:Integer.parseInt(v.get("isBusiness").toString()));
                    userDtos.add(userDto);
                }catch (Exception ex){
                    log.error("加载用户数据异常{}",v,ex);
                }
            }
        });
        return userDtos;
    }

    @Override
//    @Cacheable(key = "'id:' + #p0")
    @Transactional(rollbackFor = Exception.class)
    public UserDto findById(long id) {
        User user = userMapper.selectById(id);
        ValidationUtil.isNull(user.getId(), "User", "id", id);
        return ConvertUtil.convert(user, UserDto.class);
    }

    @Override
    public User findByJobId(Long jobId) {
        if(jobId!=null&& jobId>0) {
            return userMapper.findByJobId(jobId);
        }
        return null;
    }

    @Override
    public List<User> findListByJobId(Long jobId) {
        if(jobId!=null&& jobId>0) {
            return userMapper.findListByJobId(jobId);
        }
        return null;
    }

    @Override
    public List<User> findAll(List<Long> userId) {
        QueryWrapper<User> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("user_id", userId);
        return userMapper.selectList(queryWrapper);
    }




    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(User resources) {
        if (userMapper.findByUsername(resources.getUsername()) != null) {
            throw new EntityExistException(User.class, "用户名", resources.getUsername());
        }
//        if (userRepository.findByEmail(resources.getEmail()) != null) {
//            throw new EntityExistException(User.class, "email", resources.getEmail());
//        }
        if (userMapper.findByPhone(resources.getPhone()) != null) {
            throw new EntityExistException(User.class, "手机号码", resources.getPhone());
        }
        userMapper.insert(resources);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(User resources) {
        User user = userMapper.selectById(resources.getId());
        User user1 = userMapper.findByUsername(resources.getUsername());
        if (user1 != null && !user.getId().equals(user1.getId())) {
            throw new EntityExistException(User.class, "用户名", resources.getUsername());
        }
        // 如果用户的角色改变
        if (resources.getRoles()!=null&&!resources.getRoles().equals(user.getRoles())) {
            redisUtils.del(CacheKey.DATE_USER + resources.getId());
            redisUtils.del(CacheKey.MENU_USER + resources.getId());
            redisUtils.del(CacheKey.ROLE_AUTH + resources.getId());
        }
        // 如果用户名称修改
        if(!resources.getUsername().equals(user.getUsername())){
            redisUtils.del("user::username:" + user.getUsername());
        }
        user.setUsername(resources.getUsername());
        user.setUserSerial(resources.getUserSerial());
        user.setEmail(resources.getEmail());
        user.setEnabled(resources.getEnabled());
        user.setRoles(resources.getRoles());
        user.setDept(resources.getDept());
        user.setJobs(resources.getJobs());
        user.setPhone(resources.getPhone());
        user.setNickName(resources.getNickName());
        user.setGender(resources.getGender());
        user.setJobId(resources.getJobId());
        user.setIdentityCard(resources.getIdentityCard());
        user.setPhone(resources.getPhone());
        user.setIsBusiness(resources.getIsBusiness());
        user.setPassword(resources.getPassword());
        if(resources.getAmount()!=null)
            user.setAmount(resources.getAmount());
        userMapper.updateById(user);
        // 清除缓存
        delCaches(user.getId(), user.getUsername());
        //更新app登录用户缓存，不能删除
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCenter(User resources) {
        User user = userMapper.selectById(resources.getId());
        user.setNickName(resources.getNickName());
        user.setPhone(resources.getPhone());
        user.setGender(resources.getGender());
        userMapper.updateById(user);
        // 清理缓存
        delCaches(user.getId(), user.getUsername());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Set<Long> ids) {
        for (Long id : ids) {
            // 清理缓存
            UserDto user = findById(id);
            delCaches(user.getId(), user.getUsername());
        }
        userMapper.deleteAllByIdIn(ids);
    }

    @Override
//    @Cacheable(key = "'username:' + #p0")
    public UserDto findByName(String userName) {
        User user = userMapper.findByUsername(userName);
        if (user == null) {
            throw new EntityNotFoundException(User.class, "name", userName);
        } else {
            return ConvertUtil.convert(user, UserDto.class);
        }
    }

    @Override
    public UserDto findByPhone(String phone) {
        User user = userMapper.findByPhone(phone);
        if (user == null) {
            throw new EntityNotFoundException(User.class, "phone", phone);
        } else {
            return ConvertUtil.convert(user, UserDto.class);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePass(String username, String pass) {
        userMapper.updatePass(username, pass, new Date());
        redisUtils.del("user::username:" + username);
        flushCache(username);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<String, String> updateAvatar(MultipartFile multipartFile) {
        User user = userMapper.findByUsername(SecurityUtils.getCurrentUsername());
        String oldPath = user.getAvatarPath();
        File file = FileUtil.upload(multipartFile, properties.getPath().getAvatar());
        user.setAvatarPath(Objects.requireNonNull(file).getPath());
        user.setAvatarName(file.getName());
        userMapper.updateById(user);
        if (StringUtils.isNotBlank(oldPath)) {
            FileUtil.del(oldPath);
        }
        @NotBlank String username = user.getUsername();
        redisUtils.del(CacheKey.USER_NAME + username);
        flushCache(username);
        return new HashMap<String, String>(1) {{
            put("avatar", file.getName());
        }};
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateEmail(String username, String email) {
        userMapper.updateEmail(username, email);
        redisUtils.del(CacheKey.USER_NAME + username);
        flushCache(username);
    }

    @Override
    public void download(List<UserDto> queryAll, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (UserDto userDTO : queryAll) {
            List<String> roles = userDTO.getRoles().stream().map(RoleSmallDto::getName).collect(Collectors.toList());
            Map<String, Object> map = new LinkedHashMap<>();
            map.put("业务id", userDTO.getId());
            map.put("用户名", userDTO.getUsername());
            map.put("角色", roles);
            map.put("编号", userDTO.getUserSerial());
            map.put("账户余额", userDTO.getAmount());
            map.put("部门", userDTO.getDept().getName());
            map.put("岗位", userDTO.getJobs().stream().map(JobSmallDto::getName).collect(Collectors.toList()));
            map.put("邮箱", userDTO.getEmail());
//            map.put("状态", userDTO.getEnabled()==null ? "启用" : "禁用");
            map.put("手机号码", userDTO.getPhone());
            map.put("修改密码的时间", userDTO.getPwdResetTime());
            map.put("创建日期", userDTO.getCreateTime());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }

    /**
     * 清理缓存
     *
     * @param id /
     */
    public void delCaches(Long id, String username) {
        redisUtils.del(CacheKey.USER_ID + id);
        redisUtils.del(CacheKey.USER_NAME + username);
        flushCache(username);
    }

    /**
     * 清理 登陆时 用户缓存信息
     *
     * @param username /
     */
    private void flushCache(String username) {
        userCacheClean.cleanUserCache(username);
    }
}
