/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package admin.rest;


import admin.annotation.Log;
import admin.modules.system.domain.SysDirectory;
import admin.modules.system.service.SysDirectoryService;
import admin.modules.system.service.dto.SysDirectoryDto;
import admin.modules.system.service.dto.SysDirectoryQueryCriteria;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
* @website https://el-admin.vip
* @author lj
* @date 2023-06-01
**/
@RestController
@RequiredArgsConstructor
@Api(tags = "SysDirectoryService管理")
@RequestMapping("/api/sysDirectory")
public class SysDirectoryController {

    private final SysDirectoryService sysDirectoryService;

    @Log("导出数据")
    @ApiOperation("导出数据")
    @GetMapping(value = "/download")
    public void download(HttpServletResponse response, SysDirectoryQueryCriteria criteria) throws IOException {
        sysDirectoryService.download(sysDirectoryService.queryAll(criteria), response);
    }

    @GetMapping
    @Log("查询SysDirectoryService")
    @ApiOperation("查询SysDirectoryService")
    public ResponseEntity<Object> query(SysDirectoryQueryCriteria criteria, Pageable pageable){
        return new ResponseEntity<>(sysDirectoryService.queryAll(criteria,pageable),HttpStatus.OK);
    }

    @PostMapping
    @Log("新增SysDirectoryService")
    @ApiOperation("新增SysDirectoryService")
    public ResponseEntity<Object> create(@Validated @RequestBody SysDirectory resources){
        return new ResponseEntity<>(sysDirectoryService.create(resources),HttpStatus.CREATED);
    }

    @PutMapping
    @Log("修改SysDirectoryService")
    @ApiOperation("修改SysDirectoryService")
    public ResponseEntity<Object> update(@Validated @RequestBody SysDirectory resources){
        sysDirectoryService.update(resources);
        return new ResponseEntity<>(HttpStatus.NO_CONTENT);
    }

    @Log("删除SysDirectoryService")
    @ApiOperation("删除SysDirectoryService")
    @DeleteMapping
    public ResponseEntity<Object> delete(@RequestBody Long[] ids) {
        sysDirectoryService.deleteAll(ids);
        return new ResponseEntity<>(HttpStatus.OK);
    }


    @Log("查询目录集合")
    @ApiOperation("查询目录集合")
    @GetMapping("/list")
    public ResponseEntity<Object> list() {
        List<SysDirectoryDto> sysDirectoryDtos = sysDirectoryService.queryAll(null);
        List<SysDirectoryDto> result = sysDirectoryDtos.stream().filter(i -> i.getParentId() == 0)
                .peek(dto -> dto.setList(getChildren(dto, sysDirectoryDtos)))
                .collect(Collectors.toList());
        Map<String,Object> data= new HashMap<>();
        data.put("result",true);
        data.put("data",result);
        return new ResponseEntity<>(data,HttpStatus.OK);
    }

    @Log("查询目录叶子节点集合")
    @ApiOperation("查询目录叶子节点集合")
    @GetMapping("/sonList")
    public ResponseEntity<Object> getSonList() {
        List<SysDirectoryDto> sysDirectoryDtos = sysDirectoryService.queryAll(null);
        List<SysDirectoryDto> result = sysDirectoryDtos.stream().filter(i -> i.getParentId() == 0)
                .peek(dto -> dto.setList(getChildren(dto, sysDirectoryDtos)))
                .collect(Collectors.toList());
        result = getSon(result);
        Map<String,Object> data= new HashMap<>();
        data.put("result",true);
        data.put("data",result);
        return new ResponseEntity<>(data,HttpStatus.OK);
    }

    private List<SysDirectoryDto> getChildren(SysDirectoryDto dto, List<SysDirectoryDto> directoryList){
        return directoryList.stream().filter(i->i.getParentId().equals(dto.getDirectoryId()))
                .peek(i-> i.setList(getChildren(i,directoryList)))
                .collect(Collectors.toList());
    }

    private List<SysDirectoryDto> getSon(List<SysDirectoryDto> dto) {
        List<SysDirectoryDto> child = new ArrayList<>();
        for (SysDirectoryDto directoryDto : dto){
            if (directoryDto.getList().size() == 0){
                child.add(directoryDto);
            }else {
                child.addAll(getSon(directoryDto.getList()));
            }
        }
        return child;
    }
}
